<?php

namespace App\Http\Controllers\Api;

use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Models\InvestmentScheme;
use App\Models\PaymentMethod;
use App\Models\PromotionalMaterial;
use App\Models\Voucher;
use App\Services\AffiliateService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class FrontController extends Controller {
    public function admin_settings( Request $request ) {
        // Get system settings
        $admin_settings = system_setting();
        $user = $request->user('sanctum');
        $isAdmin = optional( $user )->role === 'ADMIN';

        // Initialize affiliate settings
        // Add affiliate packages
        $admin_settings[ 'affiliate_packages' ] = AffiliateService::allAffiliatePackages();
        if ($isAdmin) {
            $admin_settings['vouchers'] = Voucher::with(['owner:id,first_name,last_name', 'investmentScheme'])->get();
        }

        if ($user) {
            $admin_settings['vouchers_useable_by_me'] = Voucher::getUseableBy($user);
        }
        $admin_settings['investment_schemes'] = InvestmentScheme::with([
            'scheme_active_promo' => function ($q) use ($user) {
                if ($user) {
                    $q->forAudience($user->getAudienceGroups())->get();
                }
            },
            'promos',
            ...($request->isAdmin ? ['vouchers.owner'] : []),
        ])
        ->when(!$request->isAdmin, function ($query) {
            $query->where('status', StatusEnum::ACTIVE);
        })
        ->get();

        // Retrieve all payment methods
        $admin_settings['payment_methods'] = PaymentMethod::all();

        // Optimize promotional materials mapping
        $user = $request->user('sanctum');
        $admin_settings['promotional_materials'] = PromotionalMaterial::all()->map(function ($item) use ($user) {
            return [
                ...$item->toArray(),
                'meta_link' => url("api/ref?mid=".$item->id."&ref=".optional($user)->referral_code)
            ];
        });

        // If user is not an admin, remove settings starting with 'smtp' and 'mail'
        if (!$request->isAdmin) {
            $admin_settings = $this->removeSensitiveSettings($admin_settings);
        }

        // Return the optimized settings
        return $this->success($admin_settings);
    }

    /**
     * Remove settings starting with 'smtp' and 'mail' for non-admin users.
     */
    private function removeSensitiveSettings($admin_settings) {
        foreach ($admin_settings as $key => $value) {
            if (strpos($key, 'smtp') === 0 || strpos($key, 'mail' ) === 0 ) {
                        unset( $admin_settings[ $key ] );
                    }
                }

                return $admin_settings;
            }
        }
