<?php

namespace App\Http\Controllers\Api;

use App\Enums\InterestPlan;
use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Http\Requests\InvestmentSchemeRequest;
use App\Models\InvestmentScheme;
use App\Models\PaymentMethod;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class InvestmentSchemeController extends Controller {

    /**
    * Display a listing of the resource.
    */

    public function index() {
        $schemes = [];
        $schemes[ 'schemes' ] = InvestmentScheme::where( 'status', StatusEnum::ACTIVE )
        ->with( 'scheme_active_promo' )
        ->get();
        $schemes[ 'payment_methods' ] = PaymentMethod::all();
        return $this->success( $schemes );
    }

    /**
    * Store a newly created resource in storage.
    */

    public function store( InvestmentSchemeRequest $request ) {
        try {
            $validated = $request->validated();
            $validated[ 'sid' ] = 'SID' . Str::upper( Str::random( 6 ) );
            $validated[ 'interest_plans' ] = [ InterestPlan::COMPOUNDING, InterestPlan::FLEXIBLE ];
            $scheme = InvestmentScheme::create( $validated );
            return $this->success( $scheme, 'Scheme created successfully.', 201 );
        } catch ( \Throwable $th ) {
            return $this->error( 'Something went wrong', 500, $th->getMessage() );
        }
    }

    /**
    * Display the specified resource.
    */

    public function show(Request $request,  string $sid ) {
        $user = $request->user('sanctum');
        $scheme = InvestmentScheme::where( 'sid', $sid )->with( [
            'scheme_active_promo',
            'active_vouchers' => function( $q ) use ($user) {
                $q->forAudience( $user->getAudienceGroups() )->get();
            }
        ] )->first();
        if ( !$scheme )   return $this->error( 'Scheme not found', 404 );
        return $this->success( $scheme );
    }

    /**
    * Update the specified resource in storage.
    */

    public function update( InvestmentSchemeRequest $request, string $id ) {
        $validated = $request->validated();
        $scheme = InvestmentScheme::find( $id );
        if ( !$scheme )  return $this->error( 'Scheme not found', 404 );
        $scheme->update( $validated );
        return $this->success( $scheme, 'Scheme updated successfully.' );
    }

    /**
    * Remove the specified resource from storage.
    */

    public function destroy( string|int $id ) {
        try {
            $scheme = InvestmentScheme::findOrFail( $id );
            $scheme->delete();
            return $this->success( [], 'Investment Scheme deleted successfully.' );
        } catch ( \Throwable $th ) {
            //throw $th;
            return $this->error( [], 'Something went wrong.: '.$th->getMessage() );
        }
    }
}
