<?php

namespace App\Http\Controllers\Api;

use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Services\RobotRentalService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class RobotRentalController extends Controller {

    public function getRentals() {
        try {
            $rentals = RobotRentalService::getRentals();
            return $this->success( $rentals );
        } catch ( \Throwable $th ) {
            return $this->error( $th->getMessage(), 500 );
        }
    }

    /**
    * Create a new robot
    */

    public function createRobot( Request $request ): JsonResponse {
        $validatedData = $request->validate( [
            'name' => 'required|string|max:255',
            'renting_cost_per_hour' => 'required|numeric|min:0',
            'working_hours' => 'required|integer|min:1|min:1',
            'maintenance_interval_days' => 'required|integer|min:1',
            'ideal_use_case' => 'required|string|max:500',
            'media_file' => 'sometimes|required|file|mimes:jpg,jpeg,png,mp4',
            'total_robots' => 'required|integer|min:1'
        ] );

        try {
            $robot = RobotRentalService::addRobot( $validatedData );
            return $this->success( $robot, 'Robot created successfully', 201 );
        } catch ( \Throwable $th ) {
            return $this->error( $th->getMessage(), 500 );
        }
    }

    /**
    * Update a robot by its ID
    */

    public function updateRobot( Request $request, int $id ): JsonResponse {
        $validatedData = $request->validate( [
            'name' => 'sometimes|string|max:255',
            'renting_cost_per_hour' => 'sometimes|numeric|min:0',
            'working_hours' => 'sometimes|integer|min:1',
            'maintenance_interval_days' => 'sometimes|integer|min:1',
            'ideal_use_case' => 'sometimes|string|max:500',
            'media_file' => 'sometimes|required|file|mimes:jpg,jpeg,png,mp4',
            'total_robots' => 'sometimes|required|integer|min:1'
        ] );

        try {
            $updatedRobot = RobotRentalService::updateRobot( $id, $validatedData );
            return $this->success( $updatedRobot, 'Robot updated successfully' );
        } catch ( ModelNotFoundException $e ) {
            return $this->error( 'Robot not found', 404 );
        }
    }

    /**
    * Delete a robot by its ID
    */

    public function deleteRobot( int $id ): JsonResponse {
        try {
            RobotRentalService::deleteRobot( $id );
            return $this->success( 'Robot deleted successfully' );
        } catch ( ModelNotFoundException $e ) {
            return $this->error( 'Robot not found', 404 );
        }
    }

    // Company CRUD Operations

    /**
    * Create a new company
    */

    public function createCompany( Request $request ): JsonResponse {
        $validatedData = $request->validate( [
            'name' => 'required|string|max:255',
            'latitude' => 'required|string|max:255',
            'longitude' => 'required|string|max:255'
        ] );

        try {
            $company = RobotRentalService::addCompany( $validatedData );
            return $this->success( $company, 'Company created successfully', 201 );
        } catch ( ModelNotFoundException $e ) {
            return $this->error( 'Robot not found', 404 );
        }
    }

    /**
    * Update a company by its ID
    */

    public function updateCompany( Request $request, int $id ): JsonResponse {
        $validatedData = $request->validate( [
            'name' => 'sometimes|string|max:255',
            'latitude' => 'sometimes|string|max:255',
            'longitude' => 'sometimes|string|max:255'
        ] );

        try {
            $updatedCompany = RobotRentalService::updateCompany( $id, $validatedData );
            return $this->success( $updatedCompany, 'Company updated successfully' );
        } catch ( ModelNotFoundException $e ) {
            return $this->error( 'Company not found', 404 );
        }
    }

    /**
    * Delete a company by its ID
    */

    public function deleteCompany( int $id ): JsonResponse {
        try {
            RobotRentalService::deleteCompany( $id );
            return $this->success( 'Company deleted successfully' );
        } catch ( ModelNotFoundException $e ) {
            return $this->error( 'Company not found', 404 );
        }
    }

    // Rental CRUD Operations

    /**
    * Create a new rental
    */

    public function createRental( Request $request ): JsonResponse {
        $validatedData = $request->validate( [
            'company_id' => 'required|exists:companies,id',
            'robot_id' => 'required|exists:robots,id',
            'quantity' => 'required|integer|min:1',
            'rent_duration_months' => 'required|integer|min:1',
            // 'amount_per_hour' => 'required|numeric|min:0',
            'revenue_generated' => 'required|integer|min:0',
            'status' => 'required|in:'.implode( ',', StatusEnum::asArray() ),
        ] );

        try {
            $rental = RobotRentalService::addRental( $validatedData );
            return $this->success( $rental, 'Rental created successfully',  201 );
        } catch ( ModelNotFoundException $e ) {
            return $this->error( 'Robot not found', 404 );
        }
    }

    /**
    * Update a rental by its ID
    */

    public function updateRental( Request $request, int $id ): JsonResponse {
        $validatedData = $request->validate( [
            'company_id' => 'sometimes|exists:companies,id',
            'robot_id' => 'sometimes|exists:robots,id',
            'number_rented' => 'sometimes|integer|min:1',
            'rent_duration_months' => 'sometimes|integer|min:1',
            'amount_per_hour' => 'sometimes|numeric|min:0',
            'quantity' => 'sometimes|required|integer|min:1',
            'status' => 'required|in:'.implode( ',', StatusEnum::asArray() ),
        ] );

        try {
            $updatedRental = RobotRentalService::updateRental( $id, $validatedData );
            return $this->success( $updatedRental,  'Rental updated successfully' );
        } catch ( ModelNotFoundException $e ) {
            return $this->error( 'Rental not found', 404 );
        }
    }

    /**
    * Delete a rental by its ID
    */

    public function deleteRental( int $id ): JsonResponse {
        try {
            RobotRentalService::deleteRental( $id );
            return $this->success( 'Rental deleted successfully' );
        } catch ( ModelNotFoundException $e ) {
            return $this->error( 'Rental not found', 404 );
        }
    }
}
