<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class InvestmentMaturityBonusTier extends Model
{
    protected $guarded = ['id'];

    protected $casts = [
        'min_amount'       => 'decimal:2',
        'max_amount'       => 'decimal:2',
        'bonus_percentage' => 'decimal:4',
        'interest_percentage_on_investment' => 'decimal:4',
        'starts_at'        => 'datetime',
        'ends_at'          => 'datetime',
        'is_active'       => 'boolean',
    ];

    // Relationships
    public function plan()
    {
        return $this->belongsTo(InvestmentScheme::class, 'investment_scheme_id');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeCurrentlyRunning($query)
    {
        return $query->where(function ($q) {
            $q->whereNull('starts_at')
              ->orWhere('starts_at', '<=', now());
        })->where(function ($q) {
            $q->whereNull('ends_at')
              ->orWhere('ends_at', '>=', now());
        });
    }

    public function isLive(): bool
    {
        if (! $this->is_active) return false;

        $now = Carbon::now();

        if ($this->starts_at && $now->lt($this->starts_at)) return false;
        if ($this->ends_at && $now->gt($this->ends_at)) return false;

        return true;
    }

    public function range(): string
    {
        $min = '$' . number_format($this->min_amount);
        $max = $this->max_amount ? '$' . number_format($this->max_amount) : 'above';
        return "$min – $max";
    }
}
