<?php

namespace App\Models;

use App\Enums\StatusEnum;
use App\Enums\TransactionType;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use OwenIt\Auditing\Contracts\Auditable;

class Transaction extends Model implements Auditable {
    use HasFactory, \OwenIt\Auditing\Auditable;

    public $fillable = [
        'amount_usd',
        'amount_crypto',
        'user_id',
        'method_info',
        'status',
        'type',
        'expire_time'
    ];

    public $casts = [
        'method_info' => 'array',
        'expire_time' => 'timestamp'
    ];

    /**
    * Get the user associated with the transaction.
    */

    public function user() {
        return $this->belongsTo( User::class );
    }

    public function investment() {
        return $this->hasOne( Investment::class );
    }

    /**
    * Set the status of a specific transaction.
    *
    * @param int $transactionId
    * @param StatusEnum $status
    * @return \App\Models\Transaction
    * @throws \Exception
    */

    public function setTransactionStatus( string $status ): Transaction {
        if ( !StatusEnum::hasKey( $status ) ) {
            throw new \Exception( 'Invalid status' );
        }
        $investment = $this->investment;
        if ( $investment ) {
            $investment->status = StatusEnum::getValue( $status );
            $investment->save();
        }
        $this->status = StatusEnum::getValue( $status );
        $this->save();
        return $this;
    }

    public function isInitialDeposit(): bool {
        return $this->user->transactions()
        ->where( 'status', StatusEnum::APPROVED )
        ->where( 'type', TransactionType::Investment )->count() <= 1;
    }

    public function isEligibleForCommisionDeposit( int $maxCommissionDepLimit ) {
        $transactionsCount = $this->user->transactions()
        ->where( 'status', StatusEnum::APPROVED )
        ->where( 'type', TransactionType::Investment )
        ->count();

        return $transactionsCount <= $maxCommissionDepLimit;
    }

    public function depositsCount() {
        $transactionsCount = $this->user->transactions()
        ->where( 'status', StatusEnum::APPROVED )
        ->where( 'type', TransactionType::Investment )
        ->count();

        return $transactionsCount ;
    }

    public  function isSecondOrThirdDeposit(): bool {
        $transactionsCount = $this->user->transactions()
        ->where( 'status', StatusEnum::APPROVED )
        ->where( 'type', TransactionType::Investment )
        ->count();

        return $transactionsCount === 2 || $transactionsCount === 3 ;
    }

    /**
    * Get the voucher usage associated with this transaction.
    */

    public function userVoucherUsage() {
        return $this->hasOne( UserVoucherUsage::class, 'transaction_id' );
    }

    /**
    * Get the voucher associated with this transaction through the userVoucherUsage.
    */

    public function voucher() {
        return $this->hasOneThrough( Voucher::class, UserVoucherUsage::class, 'transaction_id', 'id', 'id', 'voucher_id' );
    }

    public function getSchemeAttribute() {
        return $this->investment?->scheme;
    }

}
