<?php

namespace App\Console\Commands;

use App\Enums\NotificationType;
use App\Enums\StatusEnum;
use App\Enums\TransactionType;
use App\Jobs\EmailJob;
use App\Mail\DepositComplimentEmial;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Console\Command;

class SendDepositComplimentEmail extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:send-deposit-compliment-email';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Command description';

    /**
     * Execute the console command.
     */
    public function handle() {
        $targetDate = Carbon::now()->subDays(3)->startOfDay();
        $nextDay = $targetDate->copy()->addDay();

        $users = User::with(['transactions.investment'])
            // Must have exactly 1 APPROVED deposit/investment transaction total
            ->whereHas('transactions', function ($query) {
                $query->where('status', StatusEnum::APPROVED)
                      ->whereIn('type', [TransactionType::Investment, TransactionType::Deposit]);
            }, '=', 1)
            // That one transaction must be from exactly 3 days ago
            ->whereHas('transactions', function ($query) use ($targetDate, $nextDay) {
                $query->whereBetween('updated_at', [$targetDate, $nextDay])
                      ->where('status', StatusEnum::APPROVED)
                      ->whereIn('type', [TransactionType::Investment, TransactionType::Deposit]);
            })
            // Must have at least one ACTIVE investment
            ->whereHas('transactions.investment', function ($query) {
                $query->where('status', StatusEnum::ACTIVE);
            })
            ->get();

        foreach ($users as $user) {
            EmailJob::dispatch([
                'type' => NotificationType::DepositComplimentEmail,
                'user' => $user
            ])->afterCommit();

            $this->info("User ID: {$user->id} meets the condition.");
        }
    }


}
