<?php

namespace App\Http\Controllers\Api\Auth;

use App\Http\Controllers\Controller;
use App\Enums\PinVerificationType;
use App\Mail\SendPinMail;
use App\Models\PinVerification;
use App\Models\User;
use Illuminate\Http\Request;
use BenSampo\Enum\Rules\EnumValue;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class PinVerificationController extends Controller {
    public function sendPin( Request $request ) {
        $validator = Validator::make( $request->all(), [
            'type'  => [ 'required', 'string', new EnumValue( PinVerificationType::class, true ) ],
            'email' => [ 'sometimes', 'required', 'exists:users,email' ],
        ] );

        if ( $validator->fails() ) {
            return $this->error( $validator->errors()->first(), 400, $validator->errors() );
        }

        $user = $this->findUser( $request );
        if ( !$user ) {
            return $this->error( 'Invalid email address or user not found.' );
        }

        // Optional: delete old PINs if you want one active PIN at a time
        PinVerification::where( 'user_id', $user->id )->delete();

        $pin = random_int( 100000, 999999 );

        $pinVerification = PinVerification::create( [
            'user_id'    => $user->id,
            'pin'        => Hash::make( $pin ),
            'expires_at' => now()->addMinutes( 15 ),
        ] );

        Mail::to( $user->email )->send( new SendPinMail( $pin, $request->input( 'type' ), $user ) );

        return $this->success( [
            'type'       => $request->input( 'type' ),
            'expires_at' => $pinVerification->expires_at,
        ], 'New PIN sent successfully to email.' );
    }

    public function verifyPin( Request $request ) {
        $validator = Validator::make( $request->all(), [
            'pin'   => 'required|numeric',
            'type'  => [ 'required', 'string', new EnumValue( PinVerificationType::class, true ) ],
            'email' => [ 'sometimes', 'required', 'email', 'exists:users,email' ],
        ] );

        if ( $validator->fails() ) {
            return $this->error( $validator->errors()->first(), 400, $validator->errors() );
        }

        $user = $this->findUser( $request );
        if ( !$user ) {
            return $this->error( 'Invalid email address or user not found.' );
        }

        $pinVerification = $user->pin()
        ->where( 'expires_at', '>', now() )
        ->latest( 'created_at' )
        ->first();

        if ( !$pinVerification ) {
            return $this->error( 'No valid PIN found or it has expired.' );
        }

        if ( now()->greaterThan( $pinVerification->expires_at ) ) {
            $pinVerification->delete();
            return $this->error( 'PIN expired.', 400 );
        }

        if ( !Hash::check( $request->input( 'pin' ), $pinVerification->pin ) ) {
            return $this->error( 'Invalid PIN.', 400 );
        }

        $pinVerification->delete();

        switch ( $request->input( 'type' ) ) {
            case PinVerificationType::EMAIL_VERIFICATION:
            $user->email_verified_at = now();
            $user->save();
            return $this->success( $user, 'Email verified successfully.' );

            case PinVerificationType::ACCOUNT_CHANGE:
            // Additional account change logic
            return $this->success( [], 'Account change verified successfully.' );

            case PinVerificationType::TRANSACTION:
            // Additional transaction logic
            return $this->success( [], 'Transaction verified successfully.' );

            default:
            return $this->error( 'Invalid verification type.', 400 );
        }
    }

    /**
    * Helper method to retrieve the user based on request.
    */

    private function findUser( Request $request ): ?User {
        return $request->user() ?? User::where( 'email', $request->input( 'email' ) )->first();
    }
}
