<?php

namespace App\Http\Controllers\Api\Home;

use App\Console\Commands\ProcessMaturedInvestments;
use App\Enums\InterestPlan;
use App\Enums\StatusEnum;
use App\Enums\TransactionType;
use App\Enums\UserWalletType;
use App\Http\Controllers\Controller;
use App\Http\Requests\Api\WithdrawalRequest;
use App\Http\Resources\Api\AccountResource;
use App\Models\AffiliatePackage;
use App\Models\Investment;
use App\Models\PaymentMethod;
use App\Models\SystemSettings;
use App\Models\Transaction;
use App\Services\AccountService;
use App\Services\AffiliateService;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Laravel\Fortify\Features;
use Laravel\Fortify\RecoveryCode;
use PragmaRX\Google2FA\Google2FA;

class AccountController extends Controller {

    public function user_account( Request $request ) {
        try {
            $user = $request->user();
            $user['referrals'] = AccountService::referrals();
            $user[ 'dashboard' ] = AccountService::dashboard( $user );
            $user['audits'] = AccountService::audits(limit:8);
            $user['affiliate_package'] = AffiliateService::myAffiliatePackage(user:$user);
            $user['isAdmin'] = $request->isAdmin;
            $user['my_vouchers'] = $user->vouchers()->where('user_id', $user->id)->with('investmentScheme')->get();
            $user['usable_global_vouchers'] = $user->usableGlobalVouchers();
            $user['total_invested'] = $user->transactions()
            ->where('type', TransactionType::Investment)
            ->where('status', StatusEnum::APPROVED)
            ->sum('amount_usd');
            $user['total_invested_referrals'] = $user->referredUsers()
            ->whereHas('transactions.investment', function ($query) {
                // Add any condition you need here, for example:
                // $query->where('', '>', 100);
            })->count();

            $user['two_factor_enabled'] = $user->hasEnabledTwoFactorAuthentication();
            $user['has_custom_affiliate_package'] = $user->hasCustomAffiliatPackage();

            $user['total_deposits'] = $user->transactions()
            ->whereIn( 'type', [ TransactionType::Deposit, TransactionType::Investment ] )
            ->sum( 'amount_usd' );

            $account = new AccountResource( $user );
            return $this->success( $account, 'Welcome back '.$user->first_name );
        } catch (\Throwable $th) {

        }
    }

    public function generate2FA(Request $request){
        if (!Features::enabled(Features::twoFactorAuthentication())) {
            return $this->error('2FA is not enabled', 403);
        }

        try {
            $user = $request->user();

            if ($user->hasEnabledTwoFactorAuthentication()) {
                return $this->error('2FA is already enabled', 403);
            }

            // Use Google2FA directly or your own wrapper
            $google2fa = new Google2FA();
            $secretKey = $google2fa->generateSecretKey();

            $recoveryCodes = Collection::times(8, function () {
                return RecoveryCode::generate();
            })->all();

            $user->forceFill([
                'two_factor_secret' => encrypt($secretKey),
                'two_factor_recovery_codes' => encrypt(json_encode($recoveryCodes)),
            ])->save();

            return $this->success([
                'qr_code' => $user->twoFactorQrCodeSvg(), // Assumes you have this accessor/method
                'secret_key' => $secretKey,
                'recovery_codes' => $recoveryCodes,
            ], 'Scan the QR code or enter the secret key manually in your authenticator app, then verify to enable 2FA.');

        } catch (\Exception $e) {
            Log::error('Error generating 2FA: ' . $e->getMessage(), [
                'user_id' => $request->user()?->id,
                'exception' => $e,
            ]);

            return $this->error('Failed to generate 2FA setup. Please try again later.', 500);
        }
    }


    public function enable2FA(Request $request){
        if (!Features::enabled(Features::twoFactorAuthentication())) {
            return $this->error('2FA is not enabled', 403);
        }

        $request->validate([
            'code' => 'required|numeric',
        ]);

        try {
            $user = $request->user();

            if (!$user->two_factor_secret) {
                return $this->error('2FA setup not started.', 403);
            }

            if ($user->hasEnabledTwoFactorAuthentication()) {
                return $this->error('2FA is already enabled', 403);
            }

            $google2fa = new Google2FA();
            $secret = decrypt($user->two_factor_secret);

            if (!$google2fa->verifyKey($secret, $request->code)) {
                throw ValidationException::withMessages(['code' => ['Invalid 2FA code']]);
            }

            // Mark 2FA as enabled
            $user->forceFill([
                'two_factor_confirmed_at' => now(),
            ])->save();

            return $this->success([
                'recovery_codes' => json_decode(decrypt($user->two_factor_recovery_codes), true),
            ], 'Two-factor authentication enabled');

        } catch (ValidationException $e) {
            throw $e; // Let validation errors propagate
        } catch (\Exception $e) {
            Log::error('2FA enabling failed: ' . $e->getMessage(), [
                'user_id' => $request->user()?->id,
                'exception' => $e,
            ]);
            return $this->error('Something went wrong while enabling 2FA.', 500);
        }
    }

    public function verify2FA(Request $request) {
        if (!Features::enabled(Features::twoFactorAuthentication())) {
            return $this->error('2FA is not enabled', 403);
        }

        $request->validate([
            'code' => 'required|numeric',
        ]);

        try {
            $user = $request->user();

            if (!$user->hasEnabledTwoFactorAuthentication() || !$user->two_factor_secret) {
                return $this->error('2FA is not enabled for this user.', 403);
            }

            $google2fa = new Google2FA();
            $secret = decrypt($user->two_factor_secret);

            $isValid = $google2fa->verifyKey($secret, $request->code);

            if (!$isValid) {
                throw ValidationException::withMessages(['code' => ['Invalid 2FA code']]);
            }

            $user->update(['two_factor_verified_at' => Carbon::now(), 'ip' => $request->ip()]);
            $user->ip = $request->ip();

            return $this->success(['is_verified' => true], '');

        } catch (ValidationException $e) {
            throw $e; // Rethrow validation exception as-is
        } catch (\Exception $e) {
            Log::error('2FA verification error: ' . $e->getMessage(), [
                'user_id' => $request->user()?->id,
                'code' => $request->code,
                'exception' => $e
            ]);

            return $this->error('Something went wrong during 2FA verification.', 500);
        }
    }

    public function disable2FA(Request $request) {
        if (!Features::enabled(Features::twoFactorAuthentication())) {
            return $this->error('2FA is not enabled', 403);
        }

        $user = $request->user();

        if (!$user->two_factor_secret) {
            return $this->success([], '2FA is not enabled.');
        }

        // Disable 2FA
        $user->forceFill([
            'two_factor_secret' => null,
            'two_factor_recovery_codes' => null,
            'two_factor_confirmed_at' => null,
        ])->save();

        return $this->success([], 'Two-factor authentication disabled.');
    }


    public function pending_payment( Request $request ) {
        $user = $request->user();
        $pending_payment = $user->transactions()

        ->join('investments', 'transactions.id', '=', 'investments.transaction_id')
        ->join('investment_schemes', 'investments.scheme_id', '=', 'investment_schemes.id')
        ->select(
            'transactions.*',
            'investments.*',
            'investments.transaction_id',
            'investment_schemes.*'
        )
        ->whereNotNull('transactions.expire_time')
        ->where('transactions.expire_time', '>', Carbon::now())
        ->where('transactions.type', TransactionType::Investment)
        ->where('transactions.status', StatusEnum::PENDING)
        ->orderBy('transactions.expire_time', 'desc')
        ->first();
        return $this->success( $pending_payment );
    }

    public function cancel_transaction( Request $request ) {
        $user = $request->user();
        $transaction_id = $request[ 'transaction_id' ];
        $transaction = $user->transactions()
        ->where( 'id', $transaction_id )
        ->where( 'status', StatusEnum::PENDING )
        ->with( 'investment.scheme' )
        ->first();
        if ( !$transaction ) {
            return $this->error( 'Transaction not found', 404 );
        }
        try {
            $user->cancelTransaction( $transaction );
        } catch ( \Throwable $th ) {
            Log::info( $th->getMessage() );
            return $this->error( status:500 );
        }
        return $this->success([], 'Transaction cancelled successfully.' );
    }

    public function fetch_transaction( $id ) {
        $trasanction = AccountService::transaction($id);
        return $this->success($trasanction);
    }

    public function finance_history( Request $request ) {
      try {
        $finances = AccountService::finances();
        return $this->success( $finances );
      } catch (\Throwable $th) {
        //throw $th;
        Log::error(['finance_history_error' => $th->getMessage()]);
        return $this->error('');
      }
    }

    public function set_payment_waiting(Request $request)
    {
        $validated = $request->validate([
            'transaction_id' => 'required|string|exists:transactions,id',
            'transaction_hash' => 'required|string',
        ]);

        $user = $request->user();
        $transaction_id = $validated['transaction_id'];
        $transaction_hash = $validated['transaction_hash'];

        // ❗ Check if hash already exists in any transaction's method_info JSON
        $exists = Transaction::where('method_info->transaction_hash', $transaction_hash)->exists();
        if ($exists) {
            return $this->error('A transaction with this hash already exists. Please verify the hash and try again.', 422);
        }

        $transaction = $user->transactions()
            ->where('id', $transaction_id)
            ->where('status', StatusEnum::PENDING)
            ->with('investment.scheme')
            ->first();

        if (!$transaction) {
            return $this->error('Transaction not found or is not pending', 404);
        }

        try {
            $user->setTransactionWaiting($transaction, $transaction_hash);
            return $this->success([], 'Transaction is now waiting approval.');
        } catch (\Throwable $th) {
            Log::error('Error setting transaction to waiting: ' . $th->getMessage());
            return $this->error('An error occurred while processing the transaction', 500);
        }
    }


    public function close_investment(Request $request) {

        $validated = $request->validate([
            'id' => 'required|integer|exists:investments,id',
        ]);

        $investmentId = $validated['id'];

        // Critical: Find investment and ensure it belongs to the authenticated user
        $user = $request->user();
        $isAdmin = optional($user)->role === 'ADMIN';

        $investment = Investment::query()
            ->when(! $isAdmin, function ($query) use ($user) {
                $query->whereHas('transaction', function ($q) use ($user) {
                    $q->where('user_id', $user->id);
                });
            })
            ->findOrFail($investmentId);
        // Prevent closing already closed/cancelled investments
        if (!in_array($investment->status, ['ACTIVE', 'COMPLETED'])) {
            return $this->error('This investment cannot be closed.', 400);
        }

        // Prevent double-closing
        if ($investment->status === 'COMPLETED' && $investment->closed_at !== null) {
            return $this->error('This investment is already closed.', 400);
        }

        try {
            DB::beginTransaction();

            // Use the correct total amount based on interest plan
            $returnAmount = $investment->interest_plan === 'COMPOUNDING'
                ? $investment->total_returns // full compounded amount
                : $investment->amount_invested; // only principal (interest already paid)

                $appliedBonusTier = $investment->appliedBonusTier;
                if($appliedBonusTier){
                    if($appliedBonusTier->max_amount > optional($investment->transaction->method_info)['initial_capital']){
                        $returnAmount += ($appliedBonusTier->bonus_percentage / 100) * $returnAmount;
                    }
                }

            // Call your completion logic
            ProcessMaturedInvestments::completeInvestment($investment, 0.00);

            DB::commit();

            return $this->success([
                'message' => 'Investment closed successfully!',
                'returned_amount' => $returnAmount,
                'interest_plan' => $investment->interest_plan,
            ], 'Your funds have been transferred to your wallet.');

        } catch (\Throwable $th) {
            DB::rollBack();
            Log::error('Close investment failed', [
                'user_id' => $user->id,
                'investment_id' => $investmentId,
                'error' => $th->getMessage(),
                'trace' => $th->getTraceAsString()
            ]);

            return $this->error('Failed to close investment. Please try again later.', 500);
        }
    }


    public function set_transaction_payment_method(Request $request, $transaction_id) {
        $validated = $request->validate([
            'payment_method_id' => 'required|integer|exists:payment_methods,id',
        ]);

        $user = $request->user();

        // Find the transaction that belongs to the user
        $transaction = $user->transactions()
            ->where('id', $transaction_id)
            ->first();

        // Find the payment method
        $paymentMethod = PaymentMethod::find($validated['payment_method_id']);

        if (!$transaction) {
            return $this->error('Transaction not found or does not belong to the user.', 404);
        }

        if (!$paymentMethod) {
            return $this->error('Payment Method not found.', 404);
        }

        // Check if transaction is in pending status
        if ($transaction->status !== 'PENDING') {
            return $this->error('Transaction cannot be modified. Current status: ' . $transaction->status, 400);
        }

        $methodInfo = $transaction->method_info ?? [];

        // Add payment method details
        $methodInfo['payment_method_id'] = $paymentMethod->id;
        $methodInfo['address'] = $paymentMethod->address;
        $methodInfo['short_name'] = $paymentMethod->short_name;
        $methodInfo['currency'] = $paymentMethod->currency;
        $methodInfo['network'] = $paymentMethod->network;
        $methodInfo['payment_method_updated_at'] = now()->toDateTimeString();

        $transaction->method_info = $methodInfo;
        $transaction->expire_time = Carbon::now()->addMinutes( 60 );
        // $transaction->payment_method_id = $paymentMethod->id; // If you have a direct relationship
        $transaction->save();

        return $this->success([
            'transaction' => $transaction,
            'payment_method' => $paymentMethod
        ],'Payment method set successfully');
    }

    public function compound_investment( Request $request ) {
        $validated = $request->validate( [
            'transaction_id' => 'required|integer|exists:transactions,id',
            'amount_usd' => 'required|numeric|min:0',
        ] );

        $user = $request->user();
        $transaction = $user->transactions()
        ->where( 'id', $validated[ 'transaction_id' ] )
        ->first();

        if ( !$transaction ) {
            return $this->error( 'Investment not found or does not belong to the user.', 404 );
        }

        if ((float)$validated['amount_usd'] >= (float)$transaction->investment->amount_invested) {
            return $this->error('The amount to compound exceeds the available invested amount balance.', 400);
        }

        try {
            DB::transaction( function () use ( $validated, $user, $transaction) {
                $transaction->investment->decrement('amount_invested', $validated['amount_usd']);
                $user->createReInvestmentDeposit([
                    'amount_usd' => $validated['amount_usd'],
                    'amount_crypto' => 0,
                    'scheme_id' => $transaction->investment->scheme->id,
                    'interest_plan' => InterestPlan::COMPOUNDING,
                    'payment_method' =>  array_merge(
                        $transaction->method_info ?? [],
                        ['description' => "Started Compounding investment $"."{$validated['amount_usd']} on {$transaction->investment->scheme->scheme_name} plan."]
                    )
                ]);
            });
        } catch (Exception $e) {
            Log::info($e->getMessage());
            return $this->error('An error occurred during the compounding process: ', 500 , $e->getMessage());
        }
        return $this->success($transaction, "Compounding successful");
    }

    public function reinvest_investment(Request $request) {
        $user = $request->user();

        // Validate request
        $validated = $request->validate([
            'transaction_id' => 'required|integer|exists:transactions,id',
            'amount_usd' => 'required|numeric|min:0|max:' . $user->wallet_balance,
        ]);

        // Check for sufficient balance
        if ($validated['amount_usd'] > $user->wallet_balance) {
            return $this->error('Insufficient balance', 400);
        }

        // Fetch the user's transaction
        $transaction = $user->transactions()->find($validated['transaction_id']);
        if (!$transaction) {
            return $this->error('Investment not found or does not belong to the user.', 404);
        }

        // Check minimum reinvestment amount
        $minAmount = system_setting('reinvestment_min_amount_usd', 60);
        if ($validated['amount_usd'] < $minAmount) {
            return $this->error("Amount cannot be less than $minAmount", 400);
        }

        try {
            DB::transaction(function () use ($validated, $user, $transaction) {
                // Deduct amount from wallet
                $newAmount =  $validated['amount_usd'];
                $user->decrement('wallet_balance', $newAmount);

                if ($transaction->investment->status === StatusEnum::COMPLETED) {
                    // Create a new investment deposit for completed investments
                    $user->createReInvestmentDeposit([
                        'amount_usd' => $newAmount,
                        'amount_crypto' => 0,
                        'scheme_id' => $transaction->investment->scheme->id,
                        'interest_plan' => InterestPlan::FLEXIBLE,
                        'payment_method' => array_merge(
                            $transaction->method_info ?? [],
                            ['description' => "Reinvested {$newAmount} on {$transaction->investment->scheme->scheme_name} plan."]
                        ),
                    ]);
                } else {
                    // Update existing investment for active investments
                    $transaction->investment->increment('amount_invested', $newAmount);
                    $user->createReInvestmentDeposit([
                        'amount_usd' => $newAmount,
                        'amount_crypto' => 0,
                        'scheme_id' => $transaction->investment->scheme->id,
                        'interest_plan' => InterestPlan::FLEXIBLE,
                        'payment_method' =>  array_merge(
                            $transaction->method_info ?? [],
                            ['description' => "Reinvested {$newAmount} on {$transaction->investment->scheme->scheme_name} plan."]
                        ),
                    ]);

                }
            });

            return $this->success($transaction, "Transaction successful");
        } catch (Exception $e) {
            Log::error($e->getMessage());
            return $this->error('An error occurred during the process.', 500, $e->getMessage());
        }
    }

    public function cashout_affiliate_bonus(Request $request) {
        try {
            $user = $request->user();
            $userAffiliatePackage = $user->affiliatePackageOrDefault();
            $referralsCount = $user->referrals()->count();

            if ($user->referral_commission_balance <= 0) {
                return $this->error('You do not have enough referral bonus balance to withdraw.');
            }

            $minRequired = $userAffiliatePackage['benefits']['min_referrals_required_for_cashout'] ?? 1;
            if ($referralsCount < $minRequired) {
                return $this->error("You need at least {$minRequired} referrals before you can request a cashout.");
            }

            // Payment method info (could come from request or default)
            $methodInfo = [
                'payment_method' => $request->input('payment_method', 'SYSTEM'),
                'description'    => ''
            ];

            // Create transaction
            $transaction = $user->transferReferralBonusToWallet($methodInfo);
            return $this->success($transaction, 'Your referral bonus withdrawal request has been submitted and is pending approval.');
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Withdrawal request failed: ' . $e->getMessage()
            ], 500);
        }
    }

    public function request_withdrawal(WithdrawalRequest $withdrawalRequest) {

        $user = $withdrawalRequest->user();

        $amountUsd = $withdrawalRequest['amount_usd'];
        $amountCrypto = $withdrawalRequest['amount_crypto'] ?? 0.0;
        $methodInfo = $withdrawalRequest['method_info'];
        $walletFrom = $withdrawalRequest['wallet_from'];

        $hasPendingWtd = $user->transactions()
            ->where('type', TransactionType::Withdrawal)
            ->where('status', StatusEnum::PENDING)
            ->where('method_info->wallet_from', $walletFrom)
            ->count();

        $maxPendingWtds = (int) system_setting(SystemSettings::MAX_PENDING_WITHDRAWALS, 1);

        if ($hasPendingWtd >= $maxPendingWtds) {
            if ($walletFrom !== UserWalletType::PremiumInterestBalance) {
                $messages = [
                    UserWalletType::WalletBalance             => "You already have the maximum pending withdrawals from your main balance. Please wait until one is processed or cancel a request.",
                    UserWalletType::ReferralCommissionBalance            => "Withdrawals from your referral commission balance are limited. Please wait until your pending requests are approved.",
                    UserWalletType::PremiumInterestBalance  => "Withdrawals from your premium interest balance are limited. Please wait until your pending requests are approved.",
                ];

                $message = $messages[$walletFrom] ?? "Maximum pending withdrawals limit reached. Please cancel some or wait for admin approval.";

                return $this->error($message);
            }
        }
        try {
            $transaction = $user->withdraw(
                amountUsd: $amountUsd,
                amountCrypto: $amountCrypto,
                methodInfo: $methodInfo,
                walletFrom: $walletFrom
            );

            return $this->success($transaction, 'Your withdrawal request has been submitted.');
        } catch (\Exception $e) {
            $error_message = $e->getMessage();
            // Log any errors during the withdrawal process
            Log::error('Withdrawal transaction failed', [
                'user_id' => $user->id,
                'amount_usd' => $amountUsd,
                'amount_crypto' => $amountCrypto,
                'wallet_from' => $walletFrom,
                'error_message' => $error_message,
            ]);

            return $this->error("An error occurred while processing your withdrawal request.\n\nReason: ".$error_message);
        }
    }

    public function user_account_update(Request $request){
        $user = $request->user();
        try {
            $fillable = $user->getFillable();
            $excludedFields = [ 'password', 'wallet_balance', 'email', 'role', 'status' ];
            $fieldsToUpdate = array_diff( $fillable, $excludedFields );
            $user->fill( $request->only( $fieldsToUpdate ) );
            $user->save();
            return $this->success( $user, 'Your Profile has been updated successfully.' );
        } catch ( \Throwable $th ) {
            Log::error( 'Error in AdminController -> update_user_account: ' . $th->getMessage() );
            return $this->error( 'An error occurred while updating your account.', 500, $th->getMessage() );
        }
    }

    public function join_affiliate_package(int $packageId) {
        DB::beginTransaction();
        try {
            $user = request()->user();
            $package = AffiliatePackage::findOrFail($packageId);
            if (!$package->meetsCriteriaToJoin($user)) {
                return $this->error('You do not meet the criteria to join this affiliate package.', 400);
            }
            if ($user->affiliatePackage()->exists()) {
                $user->userAffiliate()->update([
                    'affiliate_package_id' => $packageId,
                    'status' => StatusEnum::ACTIVE,
                ]);
                // return $this->error('You can only join one affiliate package at a time.', 400);
            }else{
                $user->userAffiliate()->create([
                    'affiliate_package_id' => $packageId,
                    'status' => StatusEnum::ACTIVE,
                ]);
            }
            DB::commit();
            return $this->success($user->activeAffiliatePackage, 'You have successfully joined the affiliate package.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Error joining affiliate package: " . $e->getMessage());
            return $this->error('An error occurred while joining the affiliate package.', 500);
        }
    }
}
