<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class ForceJsonResponse {
    private array $allowedOrigins = [
        'https://dash.dnxgroup.com',
        'https://www.dash.dnxgroup.com',
        'https://dnxgroup.com',
        'https://www.dnxgroup.com',
        'https://api.dnxgroup.com',
        'https://www.api.dnxgroup.com',
        'https://dash.dnx-robot.com',
        'https://www.dash.dnx-robot.com',
        'https://dnx-robot.com',
        'https://www.dnx-robot.com',
        'http://localhost:5173'
    ];

    public function handle( Request $request, Closure $next ) {
        // Force JSON response
        $request->headers->set( 'Accept', 'application/json' );

        // Determine allowed origin
        $allowedOrigin = $this->getAllowedOrigin( $request );

        // Handle OPTIONS preflight requests
        if ( $request->isMethod( 'OPTIONS' ) ) {
            return response()->noContent()
            ->withHeaders( $this->getCorsHeaders( $allowedOrigin ) );
        }

        // Process the request
        $response = $next( $request );

        // Add CORS headers to response
        return $response->withHeaders( $this->getCorsHeaders( $allowedOrigin ) );
    }

    private function getAllowedOrigin( Request $request ): string {
        // Development mode - allow all origins
        if ( !config( 'app.env' ) === 'production' ) {
            return '*';
        }

        $origin = $request->header( 'Origin' );
        $referer = $request->header( 'Referer' );

        // Case 1: Direct origin match ( most common for AJAX/fetch requests )
        if ( $origin && $this->isOriginAllowed( $origin ) ) {
            return $origin;
        }

        // Case 2: Check referer if origin is not present
        if ( !$origin && $referer ) {
            $refererOrigin = $this->extractOriginFromUrl( $referer );
            if ( $refererOrigin && $this->isOriginAllowed( $refererOrigin ) ) {
                return $refererOrigin;
            }
        }

        // Case 3: No origin/referer ( server-to-server API calls )
        if ( !$origin && !$referer ) {
            return '*';
        }

        // Log blocked request
        Log::warning( 'CORS: Blocked request', [
            'origin' => $origin ?? 'NULL',
            'referer' => $referer ?? 'NULL',
            'ip' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'path' => $request->path(),
        ] );

        // Block unauthorized origin
        abort( 403, 'Unauthorized origin' );
    }

    private function isOriginAllowed( ?string $origin ): bool {
        if ( !$origin ) {
            return false;
        }

        // Exact match
        if ( in_array( $origin, $this->allowedOrigins, true ) ) {
            return true;
        }

        // Extract domain for flexible matching
        $originHost = parse_url( $origin, PHP_URL_HOST );

        foreach ( $this->allowedOrigins as $allowed ) {
            $allowedHost = parse_url( $allowed, PHP_URL_HOST );
            if ( $originHost === $allowedHost ) {
                return true;
            }
        }

        return false;
    }

    private function extractOriginFromUrl( string $url ): ?string {
        $parsed = parse_url( $url );

        if ( !isset( $parsed[ 'scheme' ] ) || !isset( $parsed[ 'host' ] ) ) {
            return null;
        }

        $origin = $parsed[ 'scheme' ] . '://' . $parsed[ 'host' ];

        // Include port if non-standard
        if ( isset( $parsed[ 'port' ] ) ) {
            $isDefaultPort = ( $parsed[ 'scheme' ] === 'https' && $parsed[ 'port' ] === 443 ) ||
            ( $parsed[ 'scheme' ] === 'http' && $parsed[ 'port' ] === 80 );

            if ( !$isDefaultPort ) {
                $origin .= ':' . $parsed[ 'port' ];
            }
        }

        return $origin;
    }

    private function getCorsHeaders( string $allowedOrigin ): array {
        return [
            'Access-Control-Allow-Origin' => $allowedOrigin,
            'Access-Control-Allow-Methods' => 'GET, POST, PUT, PATCH, DELETE, OPTIONS',
            'Access-Control-Allow-Headers' => 'Content-Type, Authorization, X-Requested-With, X-CSRF-TOKEN, Accept',
            'Access-Control-Allow-Credentials' => 'true',
            'Access-Control-Max-Age' => '86400',
            'Vary' => 'Origin',
        ];
    }
}
