<?php

namespace App\Models;

use App\Enums\AudienceGroupEnum;
use App\Enums\StatusEnum;
use App\Enums\TransactionType;
use App\HasTransactions;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use OwenIt\Auditing\Contracts\Auditable;
use OwenIt\Auditing\Auditable as AuditableTrait;
use Laravel\Fortify\TwoFactorAuthenticatable;

class User extends Authenticatable implements MustVerifyEmail, Auditable {
    use HasFactory,
    Notifiable,
    HasApiTokens,
    HasTransactions,
    AuditableTrait,
    SoftDeletes,
    TwoFactorAuthenticatable;

    protected $fillable = [
        'first_name',
        'last_name',
        'email',
        'password',
        'wallet_balance',
        'referral_commission_balance',
        'premium_interest_balance',
        // 'role',
        'phone',
        'country',
        'bio',
        'status',
        'two_factor_verified_at',
        'ip'
    ];

    protected $hidden = [
        'password',
        'remember_token',
        'two_factor_secret',
        'two_factor_recovery_codes',
        'two_factor_verified_at',
        'ip',
    ];

    protected $guarded = [
        'role',
        'two_factor_secret',
        'two_factor_recovery_codes',
        'two_factor_confirmed_at',
        'two_factor_verified_at',
        'ip',
        'referral_commission_balance',
        'wallet_balance',
        'premium_interest_balance',
        'password',
        'status'
    ];

    protected function casts(): array {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'two_factor_verified_at' => 'datetime',
            'referral_commission_balance' => 'int',
            'premium_interest_balance' => 'int',
            'wallet_balance' => 'int'
        ];
    }

    protected $appends = [ 'invested_plans' ];

    public function getInvestedPlansAttribute() {
        $invested_scheme_names =  $this->transactions()
        ->whereHas( 'investment.scheme' ) // make sure only with scheme
        ->with( 'investment.scheme' )     // eager load to avoid N+1
        ->get()
        ->pluck( 'investment.scheme.scheme_name' )
        ->unique()
        ->values();
        return $invested_scheme_names;
    }

    public function createApiToken( string $name = 'API Access Token' ): string {
        return $this->createToken( $name )->plainTextToken;
    }

    // Relationship with referrals where the user is the referee

    public function referrals() {
        return $this->hasMany( Referral::class, 'referee_id' );
    }

    // Relationship with downlines where the user is the downline

    public function downlines() {
        return $this->hasMany( Referral::class, 'downline_id' );
    }

    // Users directly referred by this user

    public function referredUsers() {
        return $this->hasManyThrough( User::class, Referral::class, 'referee_id', 'id', 'id', 'downline_id' );
    }

    // Second-level downlines ( downlines of downlines )

    public function secondLevelDownlines() {
        return $this->hasManyThrough(
            User::class,
            Referral::class,
            'referee_id', // Foreign key on referrals table
            'id', // Foreign key on users table
            'id', // Local key on users table
            'downline_id' // Local key on referrals table
        )->with( 'downlines' );
    }

    // Method to get the profiles of referred users along with their downlines

    public function getReferredProfilesWithDownlines() {
        return $this->referredUsers()->with( [ 'downlinesProfile' ] )->get();
    }

    // Define a method to get the profile of downlines

    public function downlinesProfile() {
        return $this->hasManyThrough(
            User::class,
            Referral::class,
            'downline_id',
            'id',
            'id',
            'referee_id'
        );
    }

    public function getReferredProfilesWithDownlinesWithLevels() {
        $result = [];

        // Retrieve all first-level referrals with their downlines
        $firstLevelReferrals = $this->referrals()->with( 'downline.downlinesProfile' )->get();

        foreach ( $firstLevelReferrals as $referral ) {
            // Add the first-level referral
            $result[] = [
                'id' => $referral->downline->id,
                'name' => $referral->downline->first_name . ' ' . $referral->downline->last_name,
                'level' => 1,
                'created_at' => $referral->created_at
            ];

            // If the referral has downlines, add them
            if ( optional( $referral->downline )->downlinesProfile ) {
                foreach ( optional( $referral->downline )->downlinesProfile as $downline ) {
                    $result[] = [
                        'id' => $downline[ 'id' ],
                        'name' => $downline->first_name . ' ' . $downline->last_name,
                        'level' => 2,
                        'created_at' => $downline->created_at
                    ];
                }
            }
        }

        return $result;
    }

    // Define a method to get the upline profile through the Referral model

    public function upline() {
        return $this->hasOneThrough(
            User::class,
            Referral::class,
            'downline_id',  // Foreign key on referrals table
            'id',           // Foreign key on users table
            'id',           // Local key on users table
            'referee_id'    // Local key on referrals table
        );
    }

    public function grandUpline() {
        return $this->upline()->with( 'upline' );
    }

    /**
    * Check if a referral code is valid.
    *
    * @param  string  $code
    * @return bool
    */
    public static function isReferralCodeValid( string $code ): bool {
        return static::where( 'referral_code', $code )->exists();
    }

    /**
    * Get all investments through transactions.
    *
    * @return \Illuminate\Database\Eloquent\Relations\HasManyThrough
    */

    public function investments() {
        return $this->hasManyThrough(
            Investment::class,    // The related model
            Transaction::class,   // The intermediate model
            'user_id',            // Foreign key on the intermediate model ( Transaction )
            'transaction_id',     // Foreign key on the related model ( Investment )
            'id',                 // Local key on the parent model ( User )
            'id'                  // Local key on the intermediate model ( Transaction )
        );
    }

    // Accessor for first-level referral count

    public function getFirstLevelReferralCountAttribute() {
        return $this->referrals()->count();
    }

    public function getSecondLevelReferralCountAttribute() {
        return $this->secondLevelDownlines()->count();
    }

    public function getTotalWithdrawalsAttribute() {
        return $this->transactions()
        ->where( 'type', TransactionType::Withdrawal )
        ->sum( 'amount_usd' );
    }

    public function pin() {
        return $this->hasMany( PinVerification::class );
    }

    public function userAffiliate() {
        return $this->hasOne( UserAffiliatePackage::class );
    }

    public function affiliatePackage() {
        return $this->hasOneThrough(
            AffiliatePackage::class,
            UserAffiliatePackage::class, // Intermediate model ( UserAffiliatePackage )
            'user_id', // Foreign key on UserAffiliatePackage table
            'id', // Foreign key on AffiliatePackage table
            'id', // Local key on User table
            'affiliate_package_id' // Local key on UserAffiliatePackage table
        );
    }

    public function activeAffiliatePackage() {
        return $this->hasOneThrough(
            AffiliatePackage::class,
            UserAffiliatePackage::class, // Intermediate model ( UserAffiliatePackage )
            'user_id', // Foreign key on UserAffiliatePackage table
            'id', // Foreign key on AffiliatePackage table
            'id', // Local key on User table
            'affiliate_package_id' // Local key on UserAffiliatePackage table
        )->where( 'status', StatusEnum::ACTIVE );
    }

    // Rename the method without the 'get' prefix

    public function affiliatePackageOrDefault( ?User $user = null ) {
        $from = ( $user ?? $this );
        if ( $this->hasCustomAffiliatPackage() ) {
            $affiliatePackage =  new AffiliatePackage( $this->customAffiliatePackage->toArray() );
        } else {
            $affiliatePackage = $from->activeAffiliatePackage;
            if ( !$affiliatePackage ) {
                $affiliatePackage = AffiliatePackage::getDefaultAffiliatePackage();
            }
        }
        return $affiliatePackage;
    }

    public function customAffiliatePackage() {
        return $this->hasOne( CustomAffiliatePackage::class );
    }

    public function hasCustomAffiliatPackage(): bool {
        return $this->customAffiliatePackage &&
        $this->customAffiliatePackage->status === StatusEnum::ACTIVE;
    }

    public function vouchers(): HasMany {
        return $this->hasMany( Voucher::class );
    }

    public function usableGlobalVouchers() {
        return Voucher::globalAndUsableBy( $this->id )->get();
    }

    public function voucherUsages(): HasMany {
        return $this->hasMany( UserVoucherUsage::class );
    }

    public function getAudienceGroups(): array {
        $groups = [];

        if ( $this->hasNoInvestment() ) {
            $groups[] = AudienceGroupEnum::NoInvestment()->value;
        }

        if ( $this->isNewWithInvestment() ) {
            $groups[] = AudienceGroupEnum::NewWithInvestment()->value;
        }

        if ( $this->hasMonoPlan() ) {
            $groups[] = AudienceGroupEnum::MonoPlan()->value;
        }

        if ( $this->hasNoReferrals() ) {
            $groups[] = AudienceGroupEnum::NoReferral()->value;
        }

        return $groups;
    }

    protected function hasNoInvestment(): bool {
        return $this->investments->whereIn( 'status', [ StatusEnum::ACTIVE, StatusEnum::COMPLETED ] )->count() === 0;
    }

    protected function hasNoReferrals(): bool {
        return $this->referrals()->count() === 0;
    }

    protected function isNewWithInvestment(): bool {
        $hasInvestment = $this->investments()->count() > 0;
        $isNew = $this->created_at->gt( now()->subDays( 14 ) );
        // e.g. new = joined in last 30 days

        return $hasInvestment && $isNew;
    }

    protected function hasMonoPlan(): bool {
        return false;
    }
}
