<?php

namespace App\Observers;

use App\Enums\StatusEnum;
use App\Enums\TransactionType;
// Import the Enum
use App\Models\Transaction;
use App\Services\AffiliateService;
use App\Services\NotificationService;
use App\Models\Investment;
use Error;
use Illuminate\Support\Facades\Log;

class TransactionObserver {

    public function created( Transaction $transaction ) {
        if ( $this->shouldSkipProcessing( $transaction ) ) {
            return;
        }
        NotificationService::notifyTransactionCreatedForUser( $transaction->user,  $transaction, true );
    }

    public function updated( Transaction $transaction ) {
        if ( $this->shouldSkipProcessing( $transaction ) ) {
            return;
        }

        $transaction->load( 'investment.scheme' );

        // If not directly related, try loading manually using the embedded ID
        if ( !$transaction->investment && isset( $transaction->method_info[ 'investment_id' ] ) ) {
            $investmentId = $transaction->method_info[ 'investment_id' ];

            $investment = Investment::with( 'scheme' )->find( $investmentId );
            if ( $investment ) {
                // attach it manually for access later
                $transaction->setRelation( 'investment', $investment );
            }
        }

        try {
            if ( $this->shouldDistributeCommission( $transaction ) ) {
                AffiliateService::distributeDepositCommission( $transaction, $transaction->amount_usd );
            }
            NotificationService::notifyTransactionUpdatedForUser( $transaction->user, $transaction, true );
        } catch ( \Throwable $th ) {
            Log::error( 'Error updating transaction', [
                'message' => $th->getMessage(),
                'trace'   => $th->getTraceAsString(),
                'file'    => $th->getFile(),
                'line'    => $th->getLine(),
                'transaction_id' => $transaction->id ?? null,
            ] );

            throw new \Error( 'Error updating transaction: ' . $th->getMessage(), 0, $th );
        }
    }

    private function shouldSkipProcessing( Transaction $transaction ): bool {
        if ( optional( $transaction->method_info )[ 'is_added_by_admin' ] == true ) {
            return true;
        }
        return in_array( $transaction->type, [ TransactionType::Earnings ] );
    }

    private function shouldDistributeCommission( Transaction $transaction ): bool {
        return in_array( $transaction->type, [ TransactionType::Deposit, TransactionType::Investment, TransactionType::Topupinvestment ] ) &&
        $transaction->status == StatusEnum::APPROVED;
    }

}
