<?php

namespace App\Services;

use App\Enums\InterestPlan;
use App\Enums\StatusEnum;
use App\Enums\TransactionType;
use App\Models\Referral;
use App\Models\Transaction;
use App\Models\User;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

// use Illuminate\Support\Facades\Log;

class AccountService {
    public static function dashboard( User $user ) {
        // Cache for 2 minutes (you can adjust this duration)
        $dashboard_data = [];

        $compounding_balance = $user->transactions()
            ->where('status', StatusEnum::APPROVED)
            ->whereHas('investment', function ($query) {
                $query->where('interest_plan', InterestPlan::COMPOUNDING)
                    ->where('status', StatusEnum::ACTIVE);
            })
            ->with(['investment' => function ($query) {
                $query->where('interest_plan', InterestPlan::COMPOUNDING)
                    ->where('status', StatusEnum::ACTIVE);
            }])
            ->get()
            ->sum(function ($transaction) {
                return $transaction->investment->total_returns ?? 0;
            });

        $deposits = $user->transactions()
            ->where('status', StatusEnum::APPROVED)
            ->where('type', TransactionType::Investment)
            ->sum('amount_usd');

        $withdrawals = $user->transactions()
            ->where('status', StatusEnum::APPROVED)
            ->where('type', TransactionType::Withdrawal)
            ->sum('amount_usd');

        $referral_bonuses = $user->transactions()
            ->where('status', StatusEnum::APPROVED)
            ->where('type', TransactionType::Bonus)
            ->sum('amount_usd');

        $combined_profits_balance = $user->transactions()
            ->where('status', StatusEnum::APPROVED)
            ->where('type', TransactionType::Investment)
            ->whereHas('investment', function ($query) {
                $query->whereIn('interest_plan', [InterestPlan::COMPOUNDING, InterestPlan::FLEXIBLE])
                    ->whereIn('status', [StatusEnum::ACTIVE, StatusEnum::COMPLETED]);
            })
            ->with(['investment' => function ($query) {
                $query->whereIn('status', [StatusEnum::ACTIVE, StatusEnum::COMPLETED]);
            }])
            ->get()
            ->sum(function ($transaction) {
                return $transaction->investment->total_returns ?? 0;
            });

        $locked_balance = $user->transactions()
            ->whereIn('type', TransactionType::asArray())
            ->where('status', StatusEnum::PENDING)
            ->where(function ($query) {
                $query->whereNotNull('expire_time')
                    ->where('expire_time', '<', Carbon::now());
            })
            ->sum('amount_usd');


        $referral_profits = $user->transactions()
            ->where('status', StatusEnum::APPROVED)
            ->where('type', TransactionType::Commission)
            ->sum('amount_usd');

        $dashboard_data['wallet_balance'] = $user->wallet_balance;
        $dashboard_data['available_balance'] = $user->wallet_balance;
        $dashboard_data['locked_balance'] = $locked_balance;
        $dashboard_data['total_referrals'] = $user->referrals()->count();
        $dashboard_data['rental_profits'] = $combined_profits_balance;
        $dashboard_data['referral_profits'] = $referral_profits;
        $dashboard_data['pending_referrals'] = $user->referral_commission_balance;
        $dashboard_data['compounding_balance'] = $compounding_balance;
        $dashboard_data['deposits'] = $deposits;
        $dashboard_data['premium_interest_balance'] = $user->premium_interest_balance;
        $dashboard_data['referral_commission_balance'] = $user->referral_commission_balance;
        $dashboard_data['combined_profits_balance'] = $combined_profits_balance + $dashboard_data['referral_profits'];
        $dashboard_data['withdrawals'] = $withdrawals;
        $dashboard_data['referral_bonuses'] = $referral_bonuses;
        return $dashboard_data;
    }

    public static function transaction(int $id) {
$user = request()->user();
    $isAdmin = strtoupper($user->role) === 'ADMIN';

    // 1. First: safely find the transaction with proper ownership check
    $baseQuery = Transaction::where('transactions.id', $id); // <-- fully qualified!

    if (! $isAdmin) {
        $baseQuery->where('transactions.user_id', $user->id);
    }

    $transaction = $baseQuery->first();

    if (! $transaction) {
        abort(404, 'Transaction not found');
    }

    // 2. Now build the detailed query — always use `transactions.id`
    $query = Transaction::where('transactions.id', $id); // This is the key fix

        if ($transaction->type == TransactionType::Investment||$transaction->type == TransactionType::ReInvestment) {
            $query->select(
                'transactions.*',
                'transactions.amount_usd',
                'investment_schemes.scheme_name',
                'investments.investment_end_date',
                'investments.next_payout_time',
                'investments.status',
                'investments.id as investment_id',
                'investments.interest_plan',
                'investments.amount_invested',
                'investments.last_roi_amount',
                'investments.last_roi_date',
                'investment_schemes.sid',
                'investments.total_returns',
                'investment_schemes.min_invest_amount',
                'investment_schemes.max_invest_amount',
                'investment_schemes.type as scheme_type',

                'bonus_tiers.title as bonus_tier_title',
                'bonus_tiers.bonus_percentage as bonus_tier_percentage',
                'bonus_tiers.description as bonus_tier_description',
                'bonus_tiers.is_active as bonus_tier_is_active',

                DB::raw('JSON_UNQUOTE(JSON_EXTRACT(transactions.method_info, "$.payment_method.address")) as payment_address'),
            )
            ->join('investments', 'transactions.id', '=', 'investments.transaction_id')
            ->join('investment_schemes', 'investments.scheme_id', '=', 'investment_schemes.id')
            ->leftJoin('investment_maturity_bonus_tiers as bonus_tiers', 'investments.applied_investment_tier_id', '=', 'bonus_tiers.id');
        } elseif ($transaction->type == TransactionType::Withdrawal) {
            $query->select(
                'transactions.*',
                DB::raw('JSON_UNQUOTE(JSON_EXTRACT(transactions.method_info, "$.address")) as payment_address'),
                DB::raw('JSON_UNQUOTE(JSON_EXTRACT(transactions.method_info, "$.network")) as payment_network'),
                DB::raw('JSON_UNQUOTE(JSON_EXTRACT(transactions.method_info, "$.description")) as description')
            );
        } else {
            $query->select(
                'transactions.*',
                DB::raw('JSON_UNQUOTE(JSON_EXTRACT(transactions.method_info, "$.description")) as description')
            );
        }

        if (!$isAdmin) {
            $query->where('user_id', $user->id);
        }

        return $query->first(); // Safe now because we know it exists
    }



   public static function finances(?User $user = null) {
        $user = $user ?? request()->user();

        $pending_investment = $user->transactions()->join('investments', 'transactions.id', '=', 'investments.transaction_id')
            ->join('investment_schemes', 'investments.scheme_id', '=', 'investment_schemes.id')
            ->select(
                'transactions.*',
                'investments.*',
                'investments.transaction_id',
                'investment_schemes.*'
            )
            ->whereNotNull('transactions.expire_time')
            ->where('transactions.expire_time', '>', Carbon::now())
            ->where('transactions.type', TransactionType::Investment)
            ->where('transactions.status', StatusEnum::PENDING)
            ->orderBy('transactions.expire_time', 'desc')
            ->first();

        $investments = Transaction::select(
                'transactions.*',
                'transactions.amount_usd',
                'investment_schemes.scheme_name',
                'investments.investment_end_date',
                'investments.next_payout_time',
                'investments.status',
                'investments.id as investment_id',
                'investments.interest_plan',
                'investments.amount_invested',
                'investments.last_roi_amount',
                'investments.last_roi_date',
                'investment_schemes.sid',
                'investments.total_returns',
                'investment_schemes.min_invest_amount',
                'investment_schemes.max_invest_amount',
                'investment_schemes.type as scheme_type',

                // Bonus Tier Info
                'bonus_tiers.title as bonus_tier_title',
                'bonus_tiers.bonus_percentage as bonus_tier_percentage',
                'bonus_tiers.description as bonus_tier_description',
                'bonus_tiers.is_active as bonus_tier_is_active',

                DB::raw('JSON_UNQUOTE(JSON_EXTRACT(transactions.method_info, "$.payment_method.address")) as payment_address'),
            )
            ->join('investments', 'transactions.id', '=', 'investments.transaction_id')
            ->join('investment_schemes', 'investments.scheme_id', '=', 'investment_schemes.id')

            // LEFT JOIN because some old investments might not have a tier
            ->leftJoin('investment_maturity_bonus_tiers as bonus_tiers', 'investments.applied_investment_tier_id', '=', 'bonus_tiers.id')

            ->where('transactions.user_id', $user->id)
            ->whereIn('transactions.type', [TransactionType::Investment, TransactionType::ReInvestment])
            ->whereIn('investments.status', [StatusEnum::ACTIVE, StatusEnum::COMPLETED, StatusEnum::WAITING, StatusEnum::PENDING])
            ->orderBy('transactions.updated_at', 'DESC')
            ->orderBy('transactions.id', 'DESC')
            ->get();

        $withdrawals = Transaction::select(
            'transactions.*',
            DB::raw('JSON_UNQUOTE( JSON_EXTRACT( transactions.method_info, "$.address" ) ) as payment_address'),
            DB::raw('JSON_UNQUOTE( JSON_EXTRACT( transactions.method_info, "$.network" ) ) as payment_network')
        )
        ->where('user_id', $user->id)
        ->where('type', TransactionType::Withdrawal)
        ->orderBy('updated_at', 'DESC')
        ->orderBy('id', 'DESC')
        ->get();

        $fee_deposits = Transaction::select(
            'transactions.*',
            DB::raw('JSON_UNQUOTE( JSON_EXTRACT( transactions.method_info, "$.address" ) ) as payment_address'),
            DB::raw('JSON_UNQUOTE( JSON_EXTRACT( transactions.method_info, "$.network" ) ) as payment_network'),
        )
        ->where('user_id', $user->id)
        ->where('type', TransactionType::FeeDeposit)
        ->orderBy('updated_at', 'DESC')
        ->orderBy('id', 'DESC')
        ->get();

        // Get bonuses
        $bonuses = Transaction::select(
            'transactions.*',
            DB::raw('JSON_UNQUOTE( JSON_EXTRACT( transactions.method_info, "$.description" ) ) as description'),
        )
        ->where('user_id', $user->id)
        ->where('type', TransactionType::Bonus)
        ->orderBy('updated_at', 'DESC')
        ->orderBy('id', 'DESC')
        ->get();

        $commissions = Transaction::select(
            'transactions.*',
            DB::raw('JSON_UNQUOTE( JSON_EXTRACT( transactions.method_info, "$.description" ) ) as description'),
        )
        ->where('user_id', $user->id)
        ->where('type', TransactionType::Commission)
        ->orderBy('updated_at', 'DESC')
        ->orderBy('id', 'DESC')
        ->get();

        $earnings = Transaction::select(
            'transactions.*',
            DB::raw('JSON_UNQUOTE( JSON_EXTRACT( transactions.method_info, "$.description" ) ) as description'),
            DB::raw('JSON_UNQUOTE( JSON_EXTRACT( transactions.method_info, "$.amount_invested" ) ) as amount_invested'),
            DB::raw('JSON_UNQUOTE( JSON_EXTRACT( transactions.method_info, "$.initial_capital" ) ) as initial_capital'),
            DB::raw('JSON_UNQUOTE( JSON_EXTRACT( transactions.method_info, "$.investment_id" ) ) as investment_id'),
            DB::raw('JSON_UNQUOTE( JSON_EXTRACT( transactions.method_info, "$.scheme_name" ) ) as scheme_name'),
        )
        ->where('user_id', $user->id)
        ->where('type', TransactionType::Earnings)
        ->orderBy('updated_at', 'DESC')
        ->orderBy('id', 'DESC')
        ->get();

        $deposits_sum = $user->transactions()
            ->where('type', TransactionType::Investment)
            ->where('status', StatusEnum::APPROVED)
            ->sum('amount_usd');

        $withdrawal_sum = $user->transactions()
            ->where('type', TransactionType::Withdrawal)
            ->whereIn('status', [StatusEnum::APPROVED])
            ->sum('amount_usd');

        $bonus_sum = $user->transactions()
            ->where('type', TransactionType::Bonus)
            ->whereIn('status', [StatusEnum::APPROVED])
            ->sum('amount_usd');

        $commission_sum =  $user->transactions()
            ->where('type', TransactionType::Commission)
            ->whereIn('status', [StatusEnum::APPROVED])
            ->sum('amount_usd');

            Log::info($fee_deposits);


        // Return the transactions grouped by type
        $finances = [
            'pending_investment' =>$pending_investment,
            'commissions' => $commissions,
            'investments' => $investments,
            'withdrawals' => $withdrawals,
            'earnings' => $earnings,
            'bonuses' => $bonuses,
            'feedeposits' => $fee_deposits,
            'total_withdrawal_sum' => (int) $withdrawal_sum,
            'total_deposits_sum' => (int) $deposits_sum,
            'total_bonus_sum' => (int) $bonus_sum,
            'total_commission_sum' => (int) $commission_sum
        ];

        return $finances;
    }

    public static function referrals(?User $user = null){
        $user = $user ?? request()->user();
        $direct = Referral::where('referee_id', $user->id)
            ->with(['downline'])
            ->get()
            ->map(function ($referral) {
                $referralData = $referral->only(['id', 'referee_id', 'downline_id', 'created_at']);
                if ($referral->downline) {
                    $downline = $referral->downline;
                    $referralData['first_name'] = $downline->first_name ;
                    $referralData['last_name'] =   $downline->last_name;
                    $referralData['email'] =   $downline->email;
                    $referralData['user_id'] = $downline->id;
                }
                return $referralData;
            });


        $indirect = Referral::whereIn('referee_id', $direct->pluck('downline_id'))
            ->with(['downline'])
            ->get()
            ->map(function ($referral) {
                $referralData = $referral->only(['id', 'referee_id', 'downline_id', 'created_at']);
                $referralData['total_indrect_count'] = $referral->downline->referrals()->count();
                if ($referral->downline) {
                    $downline = $referral->downline;
                    $referralData['first_name'] = $downline->first_name ;
                    $referralData['last_name'] =   $downline->last_name;
                    $referralData['user_id'] = $downline->id;
                    $referralData['referred_by'] = $referral->referee->first_name . ' '.$referral->referee->last_name;
                }
                return $referralData;
            });

        $referrals = [
            'direct' => $direct->toArray(),
            'indirect' => $indirect->toArray(),
        ];
        return $referrals;
    }

    public static function audits(?User $user = null, $limit=9) {
        $user = $user ?? request()->user();

        $tnx_audits = Transaction::where('user_id', $user->id)
            ->with(['audits'])
            ->take($limit)
            ->orderBy( 'created_at', 'desc' )
            ->get()
            ->map(function ($transaction) {
                return [
                    'description' => $transaction->method_info['description'] ?? null,
                    'amount' => $transaction->amount_usd,
                    'status' => $transaction->status,
                    'type' => $transaction->type,
                    'created_at' => $transaction->created_at,
                ];
            })
            ->toArray();

        return $tnx_audits;
    }


}
