<?php

namespace App\Services;

use App\Enums\InvestmentSchemeType;
use App\Enums\StatusEnum;
use App\Models\User;
use App\Models\AffiliatePackage;
use App\Models\Transaction;
use Illuminate\Support\Facades\Log;

class AffiliateService {

    public static function allAffiliatePackages() {
        $allPackages[ 'list' ] = AffiliatePackage::withCount( 'userAffiliatePackages' )->get();
        $allPackages[ 'default' ] = AffiliatePackage::getDefaultAffiliatePackage();
        return $allPackages;
    }

    public static function myAffiliatePackage(?User $user = null) {
        $user = $user ? $user : request()->user('sanctum');
        if (!$user) return [];
        $myPackage = $user->affiliatePackageOrDefault();
        return $myPackage;
    }

    private static function giveCommission(User $user, float $amount, AffiliatePackage $affiliatePackage, Transaction $transaction, int $level,int $depositsCount = 0, string $commissionType, bool $isDirect, bool $isDeposit) {
        $calculateMethod = $isDeposit ? 'calculateDepositCommissionForLevel' : 'calculateProfitCommissionForLevel';
        $commission = $affiliatePackage->$calculateMethod($depositsCount, $amount, $isDirect);
        if($commission <= 0) return;
        if(!$isDeposit && !$isDirect && $level > $affiliatePackage->benefits['indirect_referral_bonus']['levels']) return;

        if($isDeposit){
            if($transaction->investment->scheme->type->value === InvestmentSchemeType::Premium ){
                $premiumDepositBenefit = $affiliatePackage->benefits['premium_deposit_benefit'] ?? 0;
                $commission += ($premiumDepositBenefit * $transaction->amount_usd) / 100;
                $commissionType =  $commissionType . " and a speacial `".$transaction->investment->scheme->scheme_name."` ";
            }
        }

        $description = $isDeposit
                ? "You've earned a $commissionType commission (Level $level) from {$transaction->user->first_name}. This is their deposit #$depositsCount."
                : "You've earned a $commissionType commission (Level $level) from {$transaction->user->first_name}.";

        // Award the commission to the user
        self::awardCommission($user, $commission, [
            'description' => $description,
            'commission_type' => $commissionType,
            'from_user_id' => $transaction->user->id,
            'from_user_name' => $transaction->user->first_name,
            'level' => $level,
            'amount' => $commission,
            'transaction_id' => $transaction->id
        ]);
    }

    private static function distributeCommission(Transaction $transaction, float $amount, bool $isDeposit) {
        $user = $transaction->user;
        $upline = $user->upline;
        if(!$upline) return;

        // Get affiliate package of the upline user
        $affiliatePackage = ($upline ?? $user)->affiliatePackageOrDefault();
        $commissionType = $isDeposit ? 'direct deposit' : 'direct Profit';

        // Get deposit bonus up to times
        $depositBonusUpToTimes = $affiliatePackage->benefits['direct_referral_bonus']['deposit_bonus_up_to_times'] ?? 0;

        // Check if the user is eligible for the deposit bonus
        if($isDeposit && !$transaction->isEligibleForCommisionDeposit($depositBonusUpToTimes)){
            return;
        }

        $depositsCount = $transaction->depositsCount();

        // Give direct commission to upline
        self::giveCommission($upline, $amount, $affiliatePackage, $transaction, 1,$depositsCount, $commissionType, true, $isDeposit);

        // Calculate indirect commissions for multiple levels
        $level = 1;
        $maxLevel = $affiliatePackage->max_referral_commission_level;
        $upline = $upline->upline;

        while ($upline && $level <= $maxLevel) {
            $affiliatePackage = $upline->affiliatePackageOrDefault();

            // Check if the upline meets the package criteria
            if ($affiliatePackage->meetsCriteriaToEarnIndirect($upline)) {
                $commissionType = $isDeposit ? 'indirect deposit' : 'indirect profit';
                self::giveCommission($upline, $amount, $affiliatePackage, $transaction, $level, $depositsCount, $commissionType, false, $isDeposit);
            }else {
                // Log::info('NO MEET CRITERIAL TO EARN INDDIRECT');
            }
            $upline = $upline->upline;
            $level++;
        }
    }

    public static function distributeDepositCommission(Transaction $transaction, float $amount) {
        $user = $transaction->user;

        // Check if the transaction is an initial deposit
        if ($transaction->isInitialDeposit()) {
            $affiliatePackage = $user->affiliatePackageOrDefault();

            // If user has upline, get upline affiliate package
            if ($user->upline) {
                $affiliatePackage = $user->upline->affiliatePackageOrDefault($user->upline);
            }

            // Calculate bonus for initial deposit
            $bonusPercentage = (float) $affiliatePackage->benefits['direct_referral_bonus']['referred_person_bonus'] ?? 100 ;
            $bonusAmount = ($bonusPercentage / 100) * $amount;

            // Award the bonus if it's valid
            if ($bonusAmount > 0) {
                $user->addBonusToInvestment(
                    $bonusAmount,
                    $transaction->investment,
                    ['description' => "🎉 You've earned a bonus of $$bonusAmount on your first deposit! 🚀", 'credit_investment' => true]
                );
            }
        }

        // Distribute the deposit commission
        self::distributeCommission($transaction, $amount, true);
    }

    public static function distributeProfitCommission(Transaction $transaction, float $amount) {
        // Distribute the profit commission
        self::distributeCommission($transaction, $amount, false);
    }

    private static function awardCommission(User $user, float $rewardAmount, array $info) {
        // Award commission to the user
        $user->addCommission($rewardAmount, $info, StatusEnum::APPROVED);
        $user->save();
    }
}
