<?php

namespace App\Services;

use App\Enums\AudienceGroupEnum;
use App\Models\Promo;
use App\Enums\PromoType;
use App\Enums\StatusEnum;
use App\Models\User;
// Assuming PromoType Enum is used
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\Log;

class PromoService {
    /**
    * Create a new promo.
    */

    public static function createPromo( array $data ): Promo {
        $promo = Promo::create( [
            'title' => optional($data)[ 'title' ],
            'description' => optional($data)[ 'description' ],
            'start_date' => optional($data)[ 'start_date' ],
            'end_date' => optional($data)[ 'end_date' ],
            'start_amount' => optional($data)[ 'start_amount' ],
            'discount_or_bonus' => optional($data)['discount_or_bonus'],
            'type' => optional($data)[ 'type' ],
            'scheme_id' => optional($data)[ 'scheme_id' ],
            'status' => optional($data)[ 'status' ] ?? StatusEnum::ACTIVE,
            'audience' => optional($data)['audience'] ?? AudienceGroupEnum::All
        ] );

        return $promo;
    }

    /**
    * Update an existing promo.
    */

    public static function updatePromo( int $promoId, array $data ): Promo {
        $promo = Promo::findOrFail( $promoId );

        $promo->update( [
            'title' => $data[ 'title' ] ?? $promo->title,
            'description' => $data[ 'description' ] ?? $promo->description,
            'start_date' => $data[ 'start_date' ] ?? $promo->start_date,
            'end_date' => $data[ 'end_date' ] ?? $promo->end_date,
            'start_amount' => $data[ 'start_amount' ] ?? $promo->start_amount,
            'type' => $data[ 'type' ] ?? $promo->type,
            'scheme_id' => $data[ 'scheme_id' ] ?? $promo->scheme_id,
            'status' => $data[ 'status' ] ?? $promo->status,
            'audience' => optional($data)['audience'] ?? AudienceGroupEnum::All
        ] );

        return $promo;
    }

    /**
    * Get all promos.
    */

    public static function getAllPromos(?User $user = null): Collection {
        return Promo::when($user, fn($q) => $q->forAudience($user->getAudienceGroups()))->get();
    }

    /**
    * Get active promos.
    */

    public static function getActivePromos(User $user): Collection {
        return Promo::where('status', 'active')
            ->where('start_date', '<=', now())
            ->where('end_date', '>=', now())
            ->forAudience($user->getAudienceGroups())
            ->get();
    }

    /**
    * Calculate the promotion reward for a **discount** promo.
    *
    * @param Promo $promo
    * @param float $purchaseAmount The total amount of the user's purchase.
    * @return float The calculated discount.
    */

    public static function calculateDiscountAmount( Promo $promo, float $purchaseAmount ): float {
        $reward = 0.0;

        // If the promo is a discount
        if ( $promo->type === PromoType::Discount ) {
            if ( $promo->start_amount > $purchaseAmount ) {
                $reward = 0.0;
            } else {
                $reward = $purchaseAmount * ( $promo->discount_or_bonus / 100 );
            }
        }

        return $reward;
    }

    /**
    * Calculate the promotion reward for a **bonus** promo.
    *
    * @param Promo $promo
    * @return float The bonus amount.
    */

    public static function calculateBonusAmount( Promo $promo, float $purchaseAmount ): float {
        $reward = 0.0;
        if ( $promo->type== PromoType::Bonus ) {
            if ( $promo->start_amount > $purchaseAmount ) {
                $reward = 0.0;
            } else {
                $reward = $purchaseAmount * ( $promo->discount_or_bonus / 100 );
            }
        }
        Log::info(['$reward' => $reward, $promo->start_amount > $purchaseAmount , $promo->start_amount , $purchaseAmount, $promo->type== PromoType::Bonus()->value , $promo->type]);
        return $reward;
    }
}
