<?php

namespace App\Services;

use App\Models\Robot;
use App\Models\Company;
use App\Models\Rental;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class RobotRentalService {

    // Add a new robot
    public static function getRentals() {
        $data[ 'robots' ] = Robot::with( [ 'rentals' ] )->get();
        $data[ 'rentals' ] = Rental::with( 'robot', 'company' )->get();
        $data[ 'companies' ] = Company::with( [ 'rentals.robot' ] )
        ->withSum( 'rentals', 'revenue_generated' )
        ->withCount( 'rentals' )
        ->orderBy( 'rentals_count', 'desc' )
        ->get();
        $data[ 'total_revenue_generated' ] = Rental::sum( 'revenue_generated' );
        $data[ 'total_rented' ] = abs( Robot::sum( 'total_rented' ) );
        $data[ 'total_maintenance_fee' ] = Robot::sum( 'total_maintenance_fee' );
        $data[ 'total_robots' ] = Robot::sum( 'total_robots' );
        $data[ 'total_available' ] = abs( $data[ 'total_robots' ]-$data[ 'total_rented' ] );
        return $data;
    }

    public static function addRobot( array $data ): Robot {
         if ( optional( $data )[ 'media_file' ] ) {
            $extension = $data[ 'media_file' ]->getClientOriginalExtension();
            $randomName = Str::random( 10 ) . '.' . $extension;
            $mediaPath = $data[ 'media_file' ]->storeAs( 'dia', $randomName, 'public' );
            $data[ 'media_url' ] =  'storage/app/public/' . $mediaPath;
        }

        return Robot::create( $data );
    }

    public static function updateRobot( int $id, array $data ): Robot {
        $robot = Robot::findOrFail( $id );
        if ( optional( $data )[ 'media_file' ] ) {
            $oldMediaPath = storage_path( "app/public/{$robot->media_url}" );
            if ( file_exists( $oldMediaPath ) && is_file( $oldMediaPath ) ) {
                unlink( $oldMediaPath );
            }

            $extension = $data[ 'media_file' ]->getClientOriginalExtension();
            $randomName = Str::random( 10 ) . '.' . $extension;
            $mediaPath = $data[ 'media_file' ]->storeAs( 'dia', $randomName, 'public' );
            $data[ 'media_url' ] =  'storage/app/public/' . $mediaPath;
        }
        $robot->update( $data );
        return $robot;
    }

    // Delete a robot

    public static function deleteRobot( int $id ): void {
        $robot = Robot::findOrFail( $id );
        $robot->delete();
    }

    // Add a new company

    public static function addCompany( array $data ): Company {
        return Company::create( $data );
    }

    // Update an existing company

    public static function updateCompany( int $id, array $data ): Company {
        $company = Company::findOrFail( $id );
        $company->update( $data );
        $company->load( 'rentals.robot' );
        return $company;
    }

    // Delete a company

    public static function deleteCompany( int $id ): void {
        $company = Company::findOrFail( $id );
        $company->delete();
    }

    // Add a new rental record

    public static function addRental( array $data ): Rental {
        $robot = Robot::findOrFail( $data[ 'robot_id' ] );
        $data[ 'amount_per_hour' ] = $robot->renting_cost_per_hour;
        if ( $data[ 'quantity' ] > $robot->total_robots ) {
            throw new Exception( 'Quantity Cannot be greather than total available robots' );
        }
        $robot->increment( 'total_rented', $data[ 'quantity' ] );
        $rental = Rental::create( $data );
        $rental->load( 'robot', 'company' );
        return $rental;
    }

    // Update an existing rental record

    public static function updateRental( int $id, array $data ): Rental {
        $rental = Rental::findOrFail( $id );
        $robot = $rental->robot;
        if ( $data[ 'quantity' ] > $robot->total_robots ) {
            throw new Exception( 'Quantity Cannot be greather than total available robots' );
        }
        if ( $data[ 'quantity' ] != $rental->quantity ) {
            if ( $data[ 'quantity' ]> $rental->quantity ) {
                $robot->decrement( 'total_rented',   $data[ 'quantity' ] );
            } else {
                $robot->increment( 'total_rented',   ( $rental->quantity-$data[ 'quantity' ] ) );
            }
        }
        $rental->update( $data );
        return $rental;
    }

    // Delete a rental record

    public static function deleteRental( int $id ): void {
        $rental = Rental::findOrFail( $id );
        $robot = $rental->robot;
        $robot->decrement( 'total_rented', $rental->quantity );
        $rental->delete();
    }

}
